%{

(*
 * m.E.M.E - Minimalist Evaluator for Mathematical Expressions
 * 
 * by Gamall Wednesday Ida
 * 
 * gamall-ida.com
 * gamall.ida@gmail.com
 * 
 * Copyright 2007, 2008 Gamall Wednesday Ida
 * 
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 *)

  (** Parser for mathematical expressions *)
  open Toolkit;;

  (** Prepare the random generator *)
  Random.self_init()

  (** The trigonometric constant *)
  let pi = 3.14159265358979323846

  (** Convert a angle in degrees to radian *)
  let deg_to_rad r = r*.(pi/.180.)

  (** Log in arbitrary base. Not too good precision *)
  let logb base x = (log x) /. (log base)
 
  (** Get the integer closest to x, or 'nearest integer' *)
  let round x = let fx = floor x in let diff = x -. fx in
    if diff <= 0.5 then fx else fx +. 1.
    
  (** Table for local variables *)
  let (local_vars : (string, float) Hashtbl.t) = Hashtbl.create 10
  
  (** Bind a local variable to a value *)
  let bind_local_var id valu = (Hashtbl.remove local_vars id; Hashtbl.add local_vars id valu)
  
  (** Get a floating-point 0 or 1 from a boolean *)
  let fob b = if b then 1. else 0.

  (** Get a bool from a float *)
  let bof f = if f = 0. then false else true
  
  (** Normalise a floating point boolean, ie get a 0 or a 1 *)
  let nfb f = if f = 0. then 0. else 1.

  (** floating point xor logical operator *)
  let fxor a b = let x, y = nfb a, nfb b in fob (x+.y = 1.)

  (** Display a binding cleanly *)
  let disp_bind k v = Printf.printf "%20s = %s\n" k (soff v) 

  (** Display all bindings *)
  let display_bindings () = 
    pl "Bound variables:"; 
    Hashtbl.iter disp_bind local_vars;
    let n = Hashtbl.length local_vars in
    pf "Total: %d %s\n" n (pluralise "variable" n)

  (** The factorial function. A classic. *) 
  let rec factorial = function
    | 0 -> 1
    | n -> assert (n > 0); n * factorial(n - 1)

  (** Factorial for floating-point numbers *)
  let factorial_f f = let n = iof f in assert (foi n = f);
    if n > 21 then 
      failwith "Factorial: Too big!" 
    else 
      foi (factorial n)

  (** Behaviour flags in case of parsing error *)
  let _se, _pe, _sl = Mathcommon._show_error, Mathcommon._print_errors, Mathcommon._show_lexbuf
  
  (** React to an error during parsing *)
  let complain_soft () = 
    let ss, se = symbol_start (), symbol_end () in
    let the_string = (Global.get Mathcommon.lexbuf).Lexing.lex_buffer in
    if !_sl && !_se && !_pe then pe the_string;
    if !_pe then begin
      (* the error message *)
      let snip = snippet (try subb the_string ss se with _ -> "BUG!") 15 in
      if se > succ ss then begin
        if !_se then Mathcommon.show_error ss se;
        epf "{Parsing failed on pos %d-%d: `%s'}\n%!" ss (pred se) snip
      end else begin
        if !_se then Mathcommon.show_error ss (ss+1); 
        let nsnip = if snip = "" then "" else va ": `%s'" snip in
        epf "{Parsing failed on pos %d%s}\n%!" ss nsnip
      end;

      (* display some help to the hapless user.... *)
      begin match snip with 
        | "=" -> pe "{Perhaps you are trying to bind a variable with a reserved name, such as `pi', `log' etc.}"
        | _ -> ()
      end;
    end
    
  (** React to error and fail *)
  let complain () = complain_soft(); failwith "Bad expression" 

  (** Get the value of a local variable *)
  let eval_local_var x = 
    try Hashtbl.find local_vars x 
    with _ -> 
      complain_soft(); 
      if !_pe then epf "{Unbound variable: `%s'}\n%!" (snippet x 15);
      failwith "Unbound variable"
%}

%token <float>  VAL 
%token <string> VAR 
%token PLUS UPLUS MINUS UMINUS TIMES DIV POW SQRT FLOOR CEIL ROUND
%token COS SIN SINH COSH TAN TANH EXP LOG ACOS ASIN ATAN LN DEG LOGB ABS MODULO FACTORIAL
%token LPAR RPAR ABSDEL LBRA RBRA LCUR RCUR
%token END
%token PI ENOT RANDOM TRUE FALSE
%token DECSEP DECXPSEP ASSIGN LET
%token LEQUALS LDIFF LSUP LINF LSSUP LSINF LAND LOR LNOT LTERN LTERNSEP LIF LELSE LTHEN LXOR
%token INCR DECR
%token CMD COMMENT
%token UNIT PRINT

%right LET DECXPSEP

%right ASSIGN
%nonassoc LTERN LTERNSEP LIF LTHEN LELSE

%left LOR LXOR
%left LAND
%left LEQUALS LDIFF
%left LSUP LINF LSSUP LSINF
%left LNOT

%nonassoc FLOOR CEIL 
%nonassoc ROUND

%left PLUS MINUS
%left TIMES DIV 
%left MODULO

%left DEG

%nonassoc LOGB
%left SQRT COS SIN SINH COSH TAN TANH EXP LOG ACOS ASIN ATAN LN ABS FACTORIAL
%left POW

%nonassoc ENOT
%nonassoc UMINUS UPLUS

%nonassoc RANDOM
%left PRINT

%start evalf
%type <float> evalf

%%

/* Entry point: A whole expression */
evalf:
  | END                         {raise Mathcommon.Unit}
  | COMMENT END                 {raise Mathcommon.Unit}
  | CMD command END             {$2}
  | xp END                      {$1}
/* Rien ne va plus mesdames et messieurs ... */
  | error                       {complain()}
;

/* A state-altering command. Mostly for debug purposes */
command:
  | VAR                         {
    begin match $1 with

    | ("vars"|"table") ->  
      display_bindings ()
    | ("clear"|"erase") ->
      Hashtbl.clear local_vars
    | ("exit"|"quit") ->
      print_endline "Bye-bye!"; exit 0
    | ("h"|"help") ->
      print_endline "Commands: vars, clear, exit."
    | _ -> 
      prerr_endline "Unrecognized command!"

    end ; raise Mathcommon.Unit 
                                }
;

/* Variable declarations/initialisations */
decs:
  | dec DECSEP decs             {$1}
  | dec                         {$1}
;

/* One variable declaration */
dec:
  | VAR ASSIGN xp               {bind_local_var $1 $3}
  | VAR INCR                    {bind_local_var $1 ((eval_local_var $1) +. 1.)}
  | INCR VAR                    {bind_local_var $2 ((eval_local_var $2) +. 1.)}
  | VAR DECR                    {bind_local_var $1 ((eval_local_var $1) -. 1.)}
  | DECR VAR                    {bind_local_var $2 ((eval_local_var $2) -. 1.)}
;

/* A delimited expression */
parxp:
  | LPAR xp RPAR                {$2}
  | LBRA xp RBRA                {$2}
  | LCUR xp RCUR                {$2}


/* Mathematical expression. The meat of the matter. */
xp:

/* Bound expressions */
  | LET decs DECXPSEP xp        {$2;$4}
  | LET decs END                {$2; raise Mathcommon.Unit}
  | decs DECXPSEP xp            {$1;$3}
  | decs END                    {$1; raise Mathcommon.Unit} 

/* Standard blocs and constants */
  | UNIT                        {raise Mathcommon.Unit}
  | VAL                         {$1}
  | parxp                       {$1}
  | VAR                         {eval_local_var $1}
  | ABSDEL xp ABSDEL            {abs_float $2}
  | PI                          {pi}
  | TRUE                        {1.}
  | FALSE                       {0.}
  | ENOT xp                     {10. ** $2}

/* Common arithmetical operators */
  | xp ENOT xp                  {$1 *. 10. ** $3}
  | xp PLUS xp                  {$1 +. $3}
  | xp MINUS xp                 {$1 -. $3}
  | xp TIMES xp                 {$1 *. $3}
  | xp DIV xp                   {$1 /. $3}
  | xp MODULO xp                {mod_float $1 $3}
  | xp POW xp                   {$1 ** $3}

/* Common mathematical library */
  | RANDOM xp                   {Random.float $2}
  | SQRT xp                     {sqrt $2}
  | FLOOR xp                    {floor $2}
  | CEIL xp                     {ceil $2}
  | ROUND xp                    {round $2}
  | ABS xp                      {abs_float $2}
  | COS xp                      {cos $2}
  | SIN xp                      {sin $2}
  | TAN xp                      {tan $2}
  | ACOS xp                     {acos $2}
  | ASIN xp                     {asin $2}
  | ATAN xp                     {atan $2}
  | COSH xp                     {cosh $2}
  | SINH xp                     {sinh $2}
  | TANH xp                     {tanh $2}
  | LN xp                       {log $2}
  | LOG xp                      {log10 $2}
  | EXP xp                      {exp $2}
  | LOGB xp parxp               {logb $2 $3}
  | DEG xp                      {deg_to_rad $2}
  | FACTORIAL xp                {factorial_f $2}

/* Logical operators and constructions */
  | xp LEQUALS xp               {fob ($1 = $3)}
  | xp LSUP xp                  {fob ($1 >= $3)}
  | xp LINF xp                  {fob ($1 <= $3)}
  | xp LSSUP xp                 {fob ($1 > $3)}
  | xp LSINF xp                 {fob ($1 < $3)}
  | xp LAND xp                  {fob (bof $1 && bof $3)}
  | xp LOR xp                   {fob (bof $1 || bof $3)}
  | xp LXOR xp                  {fxor $1 $3}
  | LNOT xp                     {if $2 = 0. then 1. else 0.}
  | xp LDIFF xp                 {fob ($1 <> $3)}
  | xp LTERN xp LTERNSEP xp     {if $1 = 0. then $5 else $3}
  | LIF xp LTHEN xp LELSE xp    {if $2 = 0. then $6 else $4}

/* Highest precedence unary operators  */
  | MINUS xp %prec UMINUS       {-.$2}
  | PLUS xp %prec UPLUS         {$2}

/* side-effects */
  | PRINT xp                    {let v= $2 in Toolkit.pl (soff v); v}
;

