{

(*
 * m.E.M.E - Minimalist Evaluator for Mathematical Expressions
 * 
 * by Gamall Wednesday Ida
 * 
 * gamall-ida.com
 * gamall.ida@gmail.com
 * 
 * Copyright 2007, 2008 Gamall Wednesday Ida
 * 
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 *)

  open Mathparser
  (** Mathematical expressions lexer  *)
} 

let floatval = ['0'-'9']+ ('.' ['0'-'9']*)? | ('.' ['0'-'9']*)

let varname  = ['a'-'z' '_'] ['-' '_' '~' '\'' 'A'-'Z' 'a' - 'z' '0'-'9'] *

let commentpat = ("(*"|"/*"|"//"|"'") _*

rule token = parse
    [' ' '\t' '\n']     { token lexbuf }     (* skip blanks *)

  | floatval as x            { VAL(float_of_string x) }
  | '@'                      { CMD }
  | commentpat               { COMMENT }
  | "()"                     { UNIT }
  | '('                      { LPAR }
  | ')'                      { RPAR }
  | '['                      { LBRA }
  | ']'                      { RBRA }
  | '{'                      { LCUR }
  | '}'                      { RCUR }
  | '|'                      { ABSDEL }
  | "++"                     { INCR }
  | "--"                     { DECR }
  | '+'                      { PLUS }
  | '-'                      { MINUS }
  | '*'                      { TIMES }
  | '/'                      { DIV }
  | "^" | "**"               { POW }
  | "floor"                  { FLOOR }
  | "ceil"                   { CEIL }
  | "rand" "om"? | '#'       { RANDOM }
  | "round"|"nint"           { ROUND }
  | "modulo"|"mod"|"%"       { MODULO } 
  | "abs"                    { ABS }
  | "sqrt"|"r"               { SQRT}
  | "cos"                    { COS }
  | "sin"                    { SIN }
  | "tan"                    { TAN }
  | "cosh"|"ch"              { COSH }
  | "sinh"|"sh"              { SINH }
  | "tanh"|"th"              { TANH }
  | "acos"                   { ACOS }
  | "asin"                   { ASIN }
  | "atan"                   { ATAN }
  | "factorial"              { FACTORIAL }
  | 'E'|'e'                  { ENOT }
  | "exp"                    { EXP }
  | "ln"                     { LN }
  | "logb"                   { LOGB }
  | "log"                    { LOG }
  | "PI"|"Pi"|"pi"           { PI }
  | "TRUE"|"true"            { TRUE }
  | "FALSE"|"false"          { FALSE }
  | "deg"                    { DEG }
  | ','|"and"                { DECSEP }
  | "in"|";"                 { DECXPSEP }
  | "=="                     { LEQUALS }
  | "<="                     { LINF }
  | ">="                     { LSUP }
  | "<"                      { LSINF }
  | ">"                      { LSSUP }
  | "&&"                     { LAND }
  | "||"                     { LOR }
  | "<>"|"!="|"|="           { LDIFF }
  | '!'|"not"                { LNOT }
  | "^^"|"xor"               { LXOR }
  | '?'                      { LTERN }
  | ':'                      { LTERNSEP }
  | "if"                     { LIF }
  | "then"                   { LTHEN }
  | "else"                   { LELSE}
  | '='|":="|"<-"            { ASSIGN }
  | "let"                    { LET }
  | "print"                  { PRINT }
  | varname as str           { VAR (str) }
  | eof                      { END }
