
(*
 * m.E.M.E - Minimalist Evaluator for Mathematical Expressions
 * 
 * by Gamall Wednesday Ida
 * 
 * gamall-ida.com
 * gamall.ida@gmail.com
 * 
 * Copyright 2007, 2008 Gamall Wednesday Ida
 * 
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 *)

(** Mathematical calculator main module *)

open Toolkit
module Eval = Matheval

(**  must unit lines be displayed ? *)
let _disp_unit = ref true

(** The calculator can be run in different modes *)
type runmode = 
  | CMD_LINE of string list
   (** Command line mode: evaluating list of expressions, printing results on each line *)
  | STDIN       
    (** Standard input mode: idem but from stdin *)
  | REPLACE      
    (** Parse expressions inside lines, generally in a file, from stdin *)

(**  selected running mode for the program *)
let _runmode = ref STDIN

(** When printing a result, remind us of the exression ? *)
let _remind = ref false

(** Insert a newline between reminders and result ? *)
let _remind_newline = ref false

(** Insert a small string before the result ? *)
let _insert_result = ref true

(** Print unit lines ? *)
let _print_unit = ref false

(** Mathematical evaluation function *)
let eval = Eval.str

(** Behaviour flags in case of parsing error *)
let _se, _pe, _sl = Mathcommon._show_error, Mathcommon._print_errors, Mathcommon._show_lexbuf

(** Print the result of an evaluation, with respect to chosen options *)
let print_eval xp = let res = eval xp in 
if !_print_unit || res <> !Eval._str_unit then
begin
  if !_remind then begin pf "%s = " xp; if !_remind_newline then ps "\n" end;
  if !_insert_result then ps "> ";
  pl res
end

(** Evaluate a list of expressions, printing results as we go *)
let rec eval_list = function
  | [] -> ()
  | xp::l -> print_eval xp ; eval_list l

(** String displayed on stderr when end of file condition is reached when reading from stdin *)
let eof_str = "{End of file reached}"

(** Evaluate what comes on stdin *)
let eval_stdin () = try while true do
  let xp = read_line () in print_eval xp
done with End_of_file -> pe eof_str; flush stderr

(** delimiters for inline expression seek and replace *)
let _delil, _delir = ref "[[", ref "]]"

(** Look for an enclosed expression in a string, then evaluate this expression and returns the same string, but with the expression replaced by its evaluated value *)
let rec seek_and_replace dl dr s = 
  let pat = Str.regexp ("\\(.*\\)" ^ dl ^ "\\(.*\\)" ^ dr ^ "\\(.*\\)$") in
  if Str.string_match pat s 0 then
    let lp, xp, rp = Str.matched_group 1 s,  Str.matched_group 2 s,  Str.matched_group 3 s in
    seek_and_replace dl dr (lp ^ (eval xp) ^ rp)
  else
    s

(** Evaluate what comes on stdin *)
let eval_stdin_replace () = 
  let dl, dr = Str.quote !_delil, Str.quote  !_delir in
  toggle _sl;
  try while true do
    let s = read_line () in pl (seek_and_replace dl dr s)
  done with End_of_file -> pe eof_str
   
(** Must the welcome message be displayed? *)
let _disp_wel = ref true

(** Display the welcome message *)
let disp_welcome() = if !_disp_wel then pl "
** m.E.M.E. v0.1
** by Gamall Wednesday Ida
** email : gamall.ida@gmail.com
** web   : gamall-ida.com
" 
     
(** Run the program according to the selected mode and options *)
let run () = match !_runmode with
  | CMD_LINE (l) -> toggle _sl; eval_list l
  | STDIN -> disp_welcome (); eval_stdin ()
  | REPLACE -> eval_stdin_replace ()

(** Display the help page *)
let disp_help () = pl "m.E.M.E - Minimalist Evaluator for Mathematical Expressions

usage: calc [options] [mode]

  modes:
    +command-line (c) sequentially evaluate command-line arguments
    +stdin        (s) evaluate expressions from standard input
    +replace      (r) search and replace expressions inside a line (on stdin)

  options:  (modes c and s)
    --print-xp     (pxp) reprint the unevaluated expression
    --print-newline (pn) goto newline before printing the result
    --print-unit    (pu) print units, ie. stand-alone assignments
    --print-symbol  (ps) print a symbol before the result

  options (mode s)
    --display-welcome (dw) 
      Display the welcome message at beginning of session

  options (modes s and r)
    --eval-now (en) xp1...xpN
      Evaluate a list of command-line expressions before processing stdin.
    
  options (mode r)
    --delimiters (del) <left del> <right del>
      Set the delimiters for mathematical expressions.

  options  (all modes)
    --eval-file    (ef) <str>
      Silently parse a file. Useful to define a bunch of variables.
    --unit-string  (us) <str>
      Set the string representing a unit (pure side-effect evaluation)
    --print-errors (pe)
      The parser prints debug error messages on stderr
    --show-error   (se)
      The parser prints a line with ^^^ where the error is
    --help
      Print this help page.
"

(** Process the expressions in a file *)
let eval_file path =
  epf "{Evaluating file `%s'}\n%!" path; run_on_file eval path

(** Parse the command-line *)
let rec pcml = function
  | [] -> ()
  | ("--print-xp"|"pxp")::l -> toggle _remind; pcml l
  | ("--print-newline"|"pn")::l -> toggle _remind_newline; pcml l
  | ("--print-unit"|"pu")::l -> toggle _print_unit; pcml l
  | ("--print-symbol"|"ps")::l -> toggle _insert_result; pcml l
  | ("--eval-now"|"en")::l -> eval_list l
  | ("--delimiters"|"del")::dl::dr::l -> _delil := dl; _delir := dr; pcml l
  | ("--unit-string"|"us")::s::l -> Eval._str_unit := s; pcml l
  | ("--eval-file"|"ef")::path::l -> eval_file path; pcml l
  | ("--display-welcome"|"dw")::l -> toggle _disp_wel; pcml l
  | ("--show-error"|"se")::l -> toggle _se; pcml l
  | ("--print_errors"|"pe")::l -> toggle _pe; pcml l
  | ("+command-line"|"c")::l -> _runmode := CMD_LINE (l)
  | ("+stdin"|"s")::l -> _runmode := STDIN ; pcml l
  | ("+replace"|"r")::l -> _runmode := REPLACE ; pcml l
  | ("--help")::l -> disp_help (); exit 0
  | x::l -> bad_args x ; failwith "Please correct the command-line and try again... (see --help)"

(** entry point *)
let _ =  begin pcml (tl al); run(); end


