
(*
 * OCaml Toolkit
 * 
 * by Gamall Wednesday Ida
 * 
 * gamall-ida.com
 * gamall.ida@gmail.com
 * 
 * Copyright 2007, 2008 Gamall Wednesday Ida
 * 
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 *)

(** 
  Some small functions and shortcuts which I find convenient.
  
  @author Gamall Wednesday Ida
*)

(** {6 Functions about lists, strings and files } *)

(** Concatenate a list of arguments, putting spaces as separators *)
let aggreg_args al = 
  let cconc a = if a <> "" then function b -> a^" "^b else function b -> a^b in
    List.fold_left cconc "" al 

(** Trim white spaces (not written by me) *)
let rec trim s =
  let l = String.length s in
  if l=0 then s 
  else if s.[0]=' ' || s.[0]='\t' || s.[0]='\n' || s.[0]='\r' then
    trim (String.sub s 1 (l-1))
  else if s.[l-1]=' ' || s.[l-1]='\t' || s.[l-1]='\n' || s.[l-1]='\r' then
    trim (String.sub s 0 (l-1))
  else s
    
(** Tail recursive version of map *)
let map_tr f l =  let rec aux acc = function
  | [] -> List.rev acc
  | hd :: tl -> aux (f hd :: acc) tl
in aux [] l
  
(** Get rid of trailing \n *)
let rec no_newline s = let l = String.length s in match l with
  | 0 -> ""
  | _ -> if s.[l-1] = '\n' || s.[l-1] = (char_of_int 13) then no_newline (String.sub s 0 (l-1)) else s;;

(** Input a file into a list, tail-recursively (not by me) *)
let lines_of_file ?(f=fun x -> x) file = let chan  = (open_in file) in
  let rec input_lines_helper res =
  let sl = 
    try
      Some (input_line chan) 
    with
      End_of_file -> None 
  in
  match sl with
       None -> List.rev res
     | Some l -> input_lines_helper ((f l) :: res) 
  in 
  input_lines_helper [];;

(** {6 Some shortcuts} *)

(** {7 Printing } *)

(** Print line to stdout *)
let pl  = print_endline 
(** Print line to stderr *)
let pe  = prerr_endline
(** Print string to stdout *)
let ps  = print_string
(** = [Printf.printf]: formated printing to stdout  *)
let pf  = Printf.printf
(** = [Printf.sprintf]: formated printing to string *)
let spf = Printf.sprintf  
(** = [Printf.eprintf]: formated printing to stderr  *)
let epf = Printf.eprintf
(** = [Printf.fprintf]: formated printing to output channel  *)
let fpf = Printf.fprintf
(** = [spf]: formated printing to stdout  *)
let va  = spf

(** {7 Lists, strings and such }*)

(** Get the tail of list *)  
let tl = List.tl
(** Get the head of a list*)
let hd = List.hd

(** = [string_of_int] *)
let soi = string_of_int
(** = [int_of_string] *)
let ios = int_of_string

(** [ios] with debug instructions *)
let iosd debug_msg x = try int_of_string x with Failure("int_of_string") -> begin
  pf "Conversion from string to number failed on token '%s'.\nReason: \"%s\"\n" x debug_msg ; 
  failwith "Cannot continue after type conversion failure." end
  
(** [ios] with zero in case of failure *)
let iosz x = try int_of_string x with Failure("int_of_string") -> 0

(** Get length of a string *)
let strlen   = String.length
(** Get length of a list *)
let listlen  = List.length
(** Get length of an array *)
let arrlen   = Array.length
  
(** DEBUG: display codes for each char in a string *)
let break_string s = let n = strlen s in
  for i = 0 to n - 1 do pf "%d " (int_of_char s.[i]) done
   

(** {7 standard channels} *)

(** [stdin] as input channel *)
let stdinc  = Pervasives.stdin
(** [stdout]  as output channel *)
let stdoutc = Pervasives.stdout
  
(** [stdin] as a file descriptor *)
let stdinf  = Unix.stdin
(** [stdout] as a file descriptor *)
let stdoutf = Unix.stdout
  
(** {6 Unix tricks and sockets} *)

(** = [Unix.handle_unix_error] *)
let hue = Unix.handle_unix_error
  
(** Open for writing at end of file *)
let open_out_append path = 
  open_out_gen [Open_wronly; Open_creat ; Open_append; Open_text] 0o666 path


(** Get a valid port number (that is, between 0 and 65535) from a string, or fail *)    
let getport p = let res = iosd "Invalid port number" p in
  if res < 0 || res > 65535 then begin
      pf "'%d' is not a valid port number. Must be in [0, 65535]\n" res;
      failwith "Incorrect port number"; end 
  else res
     

(** DEBUG: quickly display what processus I'm in *)
let pidd s = pf "I am %s (%d)\n%!" (String.uppercase s) (Unix.getpid())
  
(** Setup a pipe with channels instead of file_descr. Optionally specify if reading is non-blocking. Blocking by default. *)
let cpipe ?(nonblock=false) () = 
  let (rdfd, wrfd) = Unix.pipe () in
  if nonblock then Unix.set_nonblock rdfd;
  let rd = Unix.in_channel_of_descr rdfd in
  let wr = Unix.out_channel_of_descr wrfd in
  (rd, wr)
    
(**
  Piped double fork, leaving no zombie process behind.
 @param nb is reading non-blocking ?
 @param f  function the son will be executing. It takes two arguments: the channels to reading from the son and writing to the son, respectively.
 @return couple of channels [(channel reading from the son, channel writing to the son)]
*)
let pdfork ?(nb=false) f = 
  let getchans() = cpipe ~nonblock:nb () in
  let srd, dwr = getchans() and drd, swr = getchans() in (* crossed pipes *)
  let close_dad ()= close_in drd; close_out dwr   in (* close the ends of the dad *)
  let close_son ()= close_in srd; close_out swr   in (* close the ends of the son *)
    match Unix.fork() with
      | 0 -> (if Unix.fork() <> 0 then () else begin close_dad(); f srd swr end); exit 0
      | n ->  ignore (Unix.waitpid [] n) ; close_son(); drd, dwr
  
(** Do an [input_line] for non-blocking channels. [None] is returned if nothing is there. *)
let peek_line ic = try Some(input_line ic) with _ -> None
  
(** Display a human-legible internet address in IP:port format*)
let string_of_sockaddr = function
  | Unix.ADDR_INET (a_ip, p) -> let ip = Unix.string_of_inet_addr a_ip in va "%s:%d" ip p
  | _ -> "???"
      
(** Build a string by repeating the same pattern n times. Useful for drawing terminal lines etc *)
let rec repeat_pattern s = function
  | n when n <= 0 -> ""
  | n -> s^(repeat_pattern s (n-1))
    
(** standard separator for terminal output: double *)
let separator_double_line  = repeat_pattern "=" 80
(** standard separator for terminal output: single *)
let separator_line         = repeat_pattern "-" 80
(** standard separator for terminal output: underscore *)
let separator_solid_line   = repeat_pattern "_" 80
  
(** {6 Command-line} *)
 
(** Array of command-line arguments *)
let av = Sys.argv
(** Number of command-line arguments *)
let ac = Array.length av
(** List of command-line arguments *)
let al = Array.to_list av    

(** Incorrect command-line arguments routine: prints all arguments, insults user, and fails.  *)
let bad_args () = begin 
  epf "Incorrect runtime arguments ! Run '%s' to see help.\n\n" av.(0);  
  let i = ref 0 in Array.iter (function x -> (epf "argument %d: <%s>\n" !i x ; i := !i+1)) av ; 
end