
(*
 * Standard Input Server 
 * 
 * by Gamall Wednesday Ida
 * 
 * gamall-ida.com
 * gamall.ida@gmail.com
 * 
 * Copyright 2007, 2008 Gamall Wednesday Ida
 * 
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 * 
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 *)

open Unix;;
open Toolkit;;

(* server port *)
let _port = ref 1337
  
(* human-readable address *)
let _fulladdr = ref "XXX"
  
(* name displayed to connecting clients *)
let _name = ref "Anonymous SIS server"
   
(* do I need to control incoming connections ? *)
let _use_users = ref false
  
(* usernames and passwords *)
type user = User of string * string 
let _users = ref ([] : user list)
  
(* is this a registered user ? *)
let validate_user name pass =
  let rec find = function
    | [] -> false
    | User(n, p)::l -> (n = name && p = pass) || find l
  in find !_users
  
(* do I send stdin to stdout without touching it ? *)
let _pipe = ref false
  
(* do I need logs ? *)
let _use_logs = ref true
  
(* path to the logs *)
let _logs_path = ref "SIS.log"
  
(* file descr to the logs... will be filled at options parsing *)
let _logs = ref (None : out_channel option)
  
(* prevent the logs from displaying anything on stdout *)
let _force_silence = ref false
  
(* should the logs be flushed on each line ? *)
let _flush_logs = ref true
  
(* write something on the logs, if there are logs *)
let log ?(silent=false) str = match !_logs with
  | Some(log) -> 
      if !_flush_logs then fpf log "%s\n%!" str else fpf log "%s\n" str;
      if not (silent || !_force_silence) then pf "--> %s\n%!" str;
  | None -> ()
    
(* log this silently -no terminal display- *)
let logsil = log ~silent:true

(* is stdin printed on stdout ?*)
let _disp = ref false
  
(* micro-seconds of waiting time *)
let _delay = ref 10000

(* ...and be sure to vote for me at the next elections... err.. wrong speech *)
let welcome= "
** Standard Input Server v0.1
** by Gamall Wednesday Ida
** email : gamall.ida@gmail.com
** web   : gamall-ida.com

"
  
(* taken directly from Unix lib... *)
let rec accept_non_intr s =
  try accept s
  with Unix_error (EINTR, _, _) -> accept_non_intr s
  
(* build teh log system *)
let build_log () = if !_use_logs then _logs := Some(open_out_append !_logs_path)
  
(* guess what, this adds another user. Yes it does. *)
let add_user n p = _users := User(n, p)::!_users
  
(* command-line help *)
let display_help () = pl "
usage: sis [options]

 -port <int>                 : set the netport of the service    (default 1337)
 -user <str name> <str pass> : add user and activates access restriction
 -name                       : name for the server, displayed to the clients
 -disp                       : stdin is displayed on stdout
 -silent                     : logs will not be displayed on stdout
 -pipe                       : pipe mode (for filter chains)
 -delay    <int>             : waiting time, in microseconds    (default 10000)
 -delaysec <int>             : the same in seconds              (default     1)
 --no-logs                   : logs will not be used at all
 --logs-path <str>           : path to the log file
 --lazy-logs                 : log writing will be buffered
 --help                      : display this help page
   
"
  
(* parse the command-line *)
let rec pcml = function
  | [] -> ()
  | ("-port"|"-p")::s::l -> _port := getport s; pcml l
  | "-user"::username::password::l -> _use_users := true ; add_user username password ; pcml l
  | "-name"::name::l -> _name := name ; pcml l
  | "-disp"::l -> _disp := true ; pcml l
  | "-pipe"::l -> _pipe := true ; _force_silence := true ;pcml l
  | "-silent"::l -> _force_silence := true ; pcml l
  | "--no-logs"::l -> _use_logs := false ; pcml l
  | "--logs-path"::s::l -> _logs_path := s ; pcml l
  | "--lazy-logs"::l -> _flush_logs := false ; pcml l
  | "-delay"::usec::l -> _delay := ios usec ; pcml l
  | "-delaysec"::sec::l -> _delay := 1000000 * (ios sec) ; pcml l
  | "--help"::l -> display_help () ; exit 0
    (* I am the second clue *)
  | _ -> bad_args () ; failwith "Please correct the command-line and try again... (see --help)"
        
(* function dealing with one client, in one process *)
let deal_with_client cic coc myaddr wic woc = 
  let sendmes s = fpf coc "%s\n%!" s in 
  let getans () = no_newline (input_line cic) in
  begin
    log (va "connect: '%10s'" myaddr);
    (* interaction with the client *)
    sendmes welcome;
    sendmes (va "--> SIS Client (%s)\n--> SIS Server '%s' (%s).\n " myaddr !_name !_fulladdr);  
    
    (* login if needed *)
    if !_use_users then begin
      sendmes "Authentification is required.\nPlease enter your username: ";
      let name = getans() in
      sendmes "Please enter your password : ";
      let pass = getans() in
      log (va "login: '%s' '%s' '%s'" name pass myaddr);
      if not (validate_user name pass) then begin
        log "login failed!";
        sendmes "Access denied.\n";
        exit 0;
      end else begin
        log "login successful";
        sendmes (va "Access granted (%s)" name);
      end    
    end else begin
      log "free access";
      sendmes "Free access.\n";
    end;
      
    (* stream the data *)
    sendmes "\nREADING DATA:";
    sendmes separator_double_line;
    while true do
      try let l = input_line wic in sendmes l 
      with _ -> log (va "disconnect: '%s'" myaddr) ; exit 0;
    done
end

(* send message to all dealers, and return the list of functionnal pipes *)
let rec send_to_all mess = function
      | [] -> []
      | oc::t -> try fpf oc "%s\n%!" mess ; oc::(send_to_all mess t) 
                 with _ -> send_to_all mess t

(* get micro-second usleep command, which is not in the Unix library for reasons which are beyond me... *)
external usleep : int -> unit = "unix_usleep"
  
(* wait for connections and input, then forks the dealers *)
let wait_for_conns sock thestdin to_listener = begin
  let pipes = ref ([]: out_channel list) in
  while true do
    (* if there is a client, create a dealer and store the channel *)
    begin try
      let (s, caller) = accept_non_intr sock in 
      let inchan = in_channel_of_descr s in
      let outchan = out_channel_of_descr s in 
      let client_addr = string_of_sockaddr caller in
      let from_deal, to_deal = pdfork false (deal_with_client inchan outchan client_addr) in
      pipes :=  to_deal::!pipes
          with _ -> () end
      ;
    (* please dont eat all ressources... *)
    usleep !_delay;
    match peek_line thestdin with
      | None -> ()
      | Some(line) -> pipes := send_to_all line !pipes
  done
  
end
  
(* create the waiter, listen to stdin and transmit it to the waiter *)
let listen_to_stdin sock = begin
  let from_waiter, to_waiter = pdfork true (wait_for_conns sock) in
  while true do    
    let s = read_line () in 
    if !_disp || !_pipe then pl s;
    output_string to_waiter (s^"\n"); flush to_waiter;
  done
end
       
(* entry point *)
let main() = begin
  pcml (tl al);
  if not !_pipe then pl welcome;
  build_log();
  logsil "Starting SIS..."; 
  let local_addr = Unix.ADDR_INET(inet_addr_any, !_port) in
  _fulladdr := string_of_sockaddr local_addr;
  let sock = socket (domain_of_sockaddr local_addr) SOCK_STREAM 0 in
  setsockopt sock SO_REUSEADDR true;
  set_nonblock sock;
  bind sock local_addr;
  listen sock 5;
  Sys.set_signal Sys.sigpipe Sys.Signal_ignore;
  
  listen_to_stdin sock; 
end
  
(* handle unix errors on main *)
let _ = hue main()


(* END OF FILE *)


